<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\Invoice;
use App\Models\Worker;
use App\Models\JobListing;
use App\Models\Application;
use App\Models\Visitor;
use App\Models\Activity;
use App\Models\EmploymentStatus;
use App\Models\ExternalAgency;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class AdminController extends Controller
{
    public function dashboard()
    {
        // Get total users count
        $totalUsers = User::count();

        // Get invoice statistics
        $invoiceStats = Invoice::select([
            DB::raw('COUNT(*) as total_invoices'),
            DB::raw('SUM(total_amount) as total_revenue'),
            DB::raw('COUNT(CASE WHEN status = "pending" THEN 1 END) as pending_invoices')
        ])->first();

        // Get worker statistics
        $totalWorkers = Worker::count();
        $workerStats = [
            'available' => Worker::where('status', Worker::STATUS_AVAILABLE)->count(),
            'processing' => Worker::where('status', Worker::STATUS_PROCESSING)->count(),
            'employee' => Worker::where('status', Worker::STATUS_EMPLOYEE)->count(),
            'back_out' => Worker::where('status', Worker::STATUS_BACK_OUT)->count(),
        ];

        // Get employment status statistics
        $employmentStats = [
            'pending' => EmploymentStatus::where('status', EmploymentStatus::STATUS_PENDING)->count(),
            'sent_nea' => EmploymentStatus::where('status', EmploymentStatus::STATUS_SENT_NEA)->count(),
            'nea_approved' => EmploymentStatus::where('status', EmploymentStatus::STATUS_NEA_APPROVED)->count(),
            'medical_fit' => EmploymentStatus::where('status', EmploymentStatus::STATUS_MEDICAL_FIT)->count(),
            'ready_submission' => EmploymentStatus::where('status', EmploymentStatus::STATUS_READY_FOR_SUBMISSION)->count(),
            'submitted_embassy' => EmploymentStatus::where('status', EmploymentStatus::STATUS_SUBMITTED_EMBASSY)->count(),
            'visa_issued' => EmploymentStatus::where('status', EmploymentStatus::STATUS_VISA_ISSUED)->count(),
            'travelled' => EmploymentStatus::where('status', EmploymentStatus::STATUS_TRAVELLED)->count(),
            'cancelled' => EmploymentStatus::where('status', EmploymentStatus::STATUS_CANCELED)->count(),
        ];

        // Get job statistics
        $activeJobs = JobListing::where('status', 'active')->count();
        $totalApplications = Application::count();

        // Get visitor statistics
        $totalVisitors = Visitor::count();

        // Get recent activities
        $recentActivities = Activity::with('causer')
            ->latest()
            ->take(5)
            ->get();

        // Get recent invoices
        $recentInvoices = Invoice::with(['externalAgency'])
            ->latest()
            ->take(5)
            ->get();

        // Get external agency statistics
        $agencyStats = ExternalAgency::withCount(['workers' => function($query) {
                $query->where('workers.status', Worker::STATUS_EMPLOYEE);
            }])
            ->withCount(['invoices' => function($query) {
                $query->where('invoices.status', Invoice::STATUS_PENDING);
            }])
            ->withSum('invoices', 'total_amount')
            ->get()
            ->map(function($agency) {
                return (object)[
                    'name' => $agency->display_name,
                    'active_workers' => $agency->workers_count,
                    'pending_invoices' => $agency->invoices_count,
                    'total_revenue' => $agency->invoices_sum_total_amount ?? 0
                ];
            });

        return view('admin.dashboard', [
            'totalUsers' => $totalUsers,
            'totalInvoices' => $invoiceStats->total_invoices ?? 0,
            'totalRevenue' => $invoiceStats->total_revenue ?? 0,
            'pendingInvoices' => $invoiceStats->pending_invoices ?? 0,
            'totalWorkers' => $totalWorkers,
            'workerStats' => $workerStats,
            'employmentStats' => $employmentStats,
            'activeJobs' => $activeJobs,
            'totalApplications' => $totalApplications,
            'totalVisitors' => $totalVisitors,
            'recentActivities' => $recentActivities,
            'recentInvoices' => $recentInvoices,
            'agencyStats' => $agencyStats
        ]);
    }
}