<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Spatie\Backup\BackupDestination\Backup;
use Spatie\Backup\BackupDestination\BackupDestination;
use Spatie\Backup\Tasks\Backup\BackupJob;
use Spatie\Backup\Tasks\Backup\BackupJobFactory;
use Illuminate\Support\Facades\Storage;
use Carbon\Carbon;
use Illuminate\Support\Facades\Artisan;

class BackupController extends Controller
{
    public function create()
    {
        return view('backup.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'backup_type' => 'required|in:full,database,files'
        ]);

        try {
            $type = $request->backup_type;
            $options = [];
            if ($type === 'database') {
                $options['--only-db'] = true;
            } elseif ($type === 'files') {
                $options['--only-files'] = true;
            }
            // Run the backup:run command
            Artisan::call('backup:run', $options);

            return redirect()->back()->with('success', 'Backup created successfully!');
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Failed to create backup: ' . $e->getMessage());
        }
    }

    public function list()
    {
        $backups = collect(Storage::disk('local')->files('backups'))
            ->filter(function ($file) {
                return pathinfo($file, PATHINFO_EXTENSION) === 'zip';
            })
            ->map(function ($file) {
                return [
                    'name' => basename($file),
                    'size' => Storage::disk('local')->size($file),
                    'created_at' => Carbon::createFromTimestamp(Storage::disk('local')->lastModified($file)),
                ];
            })
            ->sortByDesc('created_at');

        return view('backup.list', compact('backups'));
    }

    public function download($filename)
    {
        $path = 'backups/' . $filename;
        
        if (!Storage::disk('local')->exists($path)) {
            return redirect()->back()->with('error', 'Backup file not found.');
        }

        return Storage::disk('local')->download($path);
    }

    public function delete($filename)
    {
        $path = 'backups/' . $filename;
        
        if (!Storage::disk('local')->exists($path)) {
            return redirect()->back()->with('error', 'Backup file not found.');
        }

        Storage::disk('local')->delete($path);

        return redirect()->back()->with('success', 'Backup deleted successfully.');
    }
} 