<?php

namespace App\Http\Controllers;

use App\Models\EmploymentStatus;
use App\Models\ExternalAgency;
use App\Models\Worker;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\ValidationException;
use App\Exports\EmploymentStatussExport;
use Maatwebsite\Excel\Facades\Excel;

class EmploymentStatusController extends Controller
{
    public function index(Request $request)
    {
        $query = EmploymentStatus::query()
            ->with(['worker', 'externalAgency'])
            ->join('workers', 'employment_statuses.worker_id', '=', 'workers.id')
            ->select('employment_statuses.*', 'workers.passport_number', 'workers.given_names', 'workers.surname');

        // Filters
        if ($request->filled('passport_number')) {
            $query->where('workers.passport_number', 'like', '%' . $request->passport_number . '%');
        }

        if ($request->filled('visa_number')) {
            $query->where('employment_statuses.visa_number', 'like', '%' . $request->visa_number . '%');
        }

        if ($request->filled('status')) {
            $query->where('employment_statuses.status', $request->status);
        }

        if ($request->filled('payment_status')) {
            $query->where('employment_statuses.payment_status', $request->payment_status);
        }

        if ($request->filled('external_agency_id')) {
            $query->where('employment_statuses.external_agency_id', $request->external_agency_id);
        }

        // Sorting
        $sortBy = $request->get('sort_by', 'created_at');
        $order = $request->get('order', 'desc');
        $query->orderBy('employment_statuses.' . $sortBy, $order);

        $EmploymentStatus = $query->paginate(10)->appends($request->query());

        return view('employment_status.index', compact('EmploymentStatus'));
    }

    public function export(Request $request)
    {
        return Excel::download(new EmploymentStatussExport($request->all()), 'employment_requests.xlsx');
    }

    public function create(Request $request)
    {
        $worker = null;
        $workerNotAvailable = false;

        if ($request->passport_number) {
            $worker = Worker::where('passport_number', $request->passport_number)->first();

            if ($worker && $worker->status !== 'available') {
                $workerNotAvailable = true;
            }
        }

        $lastContractNumber = EmploymentStatus::max('contract_number');
        $nextNumber = 1;

        if ($lastContractNumber && preg_match('/CNT-(\d+)/', $lastContractNumber, $matches)) {
            $nextNumber = (int) $matches[1] + 1;
        }

        $formattedContractNumber = 'CNT-' . str_pad($nextNumber, 3, '0', STR_PAD_LEFT);

        return view('employment_status.create', [
            'worker' => $worker,
            'workerNotAvailable' => $workerNotAvailable,
            'external_agencies' => ExternalAgency::all(),
            'nextContractNumber' => $formattedContractNumber,
        ]);
    }

    public function store(Request $request)
    {
        // Automatically set default values if not explicitly provided
        $request->merge([
            'status' => $request->input('status', EmploymentStatus::STATUS_PENDING),
            'payment_status' => 'Pending', // <-- Default payment_status set here
        ]);

        $validated = $request->validate([
            'worker_id'          => 'required|exists:workers,id',
            'contract_number'    => 'required|string|max:255|unique:employment_statuses|regex:/^CNT-\d{3}$/',
            'visa_number'        => 'required|string|max:255',
            'payment_status'     => 'required|in:Paid,Pending',
            'status'             => 'required|in:Pending,Sent NEA,NEA Approved,Medical Fit,Ready for Submission,Submitted Embassy,Visa Issued,Travelled',
            'creation_date'      => 'nullable|date',
            'external_agency_id' => 'nullable|exists:external_agencies,id',
        ]);

        DB::beginTransaction();

        try {
            // Get the worker to access their occupation
            $worker = Worker::findOrFail($request->worker_id);

            // Create the employment status with the worker's occupation
            $employmentStatus = EmploymentStatus::create([
                'worker_id' => $validated['worker_id'],
                'contract_number' => $validated['contract_number'],
                'visa_number' => $validated['visa_number'],
                'payment_status' => $validated['payment_status'],
                'status' => $validated['status'],
                'creation_date' => $validated['creation_date'] ?? now(),
                'external_agency_id' => $validated['external_agency_id'],
            ]);

            // Update worker status to 'processing'
            $worker->update(['status' => Worker::STATUS_PROCESSING]);

            DB::commit();

            return redirect()->route('employment_status.index')
                ->with('success', 'Employment status created successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            throw ValidationException::withMessages([
                'error' => ['Failed to create employment status. Please try again.']
            ]);
        }
    }

    public function show(EmploymentStatus $EmploymentStatus)
    {
        $EmploymentStatus->load(['worker', 'externalAgency']);

        $employment = $EmploymentStatus;
        $worker = $EmploymentStatus->worker;

        return view('employment_status.show', compact('employment', 'worker'));
    }

    public function edit(EmploymentStatus $EmploymentStatus)
    {
        $external_agencies = ExternalAgency::all();
        return view('employment_status.edit', [
            'employmentRequest' => $EmploymentStatus,
            'external_agencies' => $external_agencies
        ]);
    }

    public function update(Request $request, EmploymentStatus $EmploymentStatus)
    {
        $validated = $request->validate([
            'contract_number'    => 'required|string|max:255|unique:employment_statuses,contract_number,' . $EmploymentStatus->id,
            'visa_number'        => 'required|string|max:255',
            'payment_status'     => 'required|in:Paid,Pending',
            'status'             => 'required|in:Pending,Sent NEA,NEA Approved,Medical Fit,Ready for Submission,Submitted Embassy,Visa Issued,Travelled',
            'creation_date'      => 'nullable|date',
            'external_agency_id' => 'nullable|exists:external_agencies,id',
        ]);

        $EmploymentStatus->update($validated);

        return redirect()->route('employment_status.index')
            ->with('success', 'Employment record updated successfully.');
    }

    public function cancel(Request $request, $id)
    {
        $request->validate([
            'cancellation_reason' => 'required|string|max:1000',
        ]);

        DB::transaction(function () use ($request, $id) {
            $employment = EmploymentStatus::findOrFail($id);

            // Cancel the contract
            $employment->update([
                'status' => EmploymentStatus::STATUS_CANCELED,
                'cancellation_reason' => $request->input('cancellation_reason'),
                'cancelled_by' => auth()->id(),
            ]);

            // Update the associated worker
            if ($employment->worker) {
                $employment->worker->update([
                    'status' => 'available',
                    'current_contract_id' => null,
                ]);
            }
        });

        return back()->with('success', 'Contract canceled and worker set to available.');
    }

    public function advanceStatus($id)
    {
        try {
            DB::beginTransaction();

            $employmentStatus = EmploymentStatus::findOrFail($id);
            $currentStatus = $employmentStatus->status;
            $nextStatus = $this->getNextStatus($currentStatus);

            if (!$nextStatus) {
                return back()->with('error', [
                    'title' => 'Error!',
                    'message' => 'Cannot advance status further.',
                    'type' => 'error'
                ]);
            }

            $employmentStatus->update(['status' => $nextStatus]);

            // Add success message with specific content for 'travelled' status
            $message = [
                'title' => 'Status Updated',
                'message' => 'Employment status has been updated to ' . $nextStatus,
                'details' => [
                    'worker' => $employmentStatus->worker->given_names . ' ' . $employmentStatus->worker->surname,
                    'contract_number' => $employmentStatus->contract_number,
                    'new_status' => $nextStatus
                ]
            ];

            if ($nextStatus === EmploymentStatus::STATUS_TRAVELLED) {
                $message['details']['worker_status'] = 'Employee';
            }

            DB::commit();

            return back()->with('success', $message);

        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Failed to advance employment status: ' . $e->getMessage());
            return back()->with('error', [
                'title' => 'Error!',
                'message' => 'Failed to update employment status. Please try again.',
                'type' => 'error'
            ]);
        }
    }

    private function getNextStatus($currentStatus)
    {
        return match($currentStatus) {
            EmploymentStatus::STATUS_PENDING => EmploymentStatus::STATUS_SENT_NEA,
            EmploymentStatus::STATUS_SENT_NEA => EmploymentStatus::STATUS_NEA_APPROVED,
            EmploymentStatus::STATUS_NEA_APPROVED => EmploymentStatus::STATUS_MEDICAL_FIT,
            EmploymentStatus::STATUS_MEDICAL_FIT => EmploymentStatus::STATUS_READY_FOR_SUBMISSION,
            EmploymentStatus::STATUS_READY_FOR_SUBMISSION => EmploymentStatus::STATUS_SUBMITTED_EMBASSY,
            EmploymentStatus::STATUS_SUBMITTED_EMBASSY => EmploymentStatus::STATUS_VISA_ISSUED,
            EmploymentStatus::STATUS_VISA_ISSUED => EmploymentStatus::STATUS_TRAVELLED,
            default => null
        };
    }

    public function invoice()
    {
        return $this->hasOne(Invoice::class, 'employment_status_id');
    }
}



