<?php

namespace App\Http\Controllers;

use App\Models\ExternalAgency;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;

class ExternalAgencyController extends Controller
{
    public function index(Request $request)
    {
        $external_agencies = ExternalAgency::query()
            // Search by responsible name if provided
            ->when($request->filled('responsible_name'), function ($query) use ($request) {
                $query->where('responsible_name', 'like', '%' . $request->responsible_name . '%');
            })
            // Search by company name if provided
            ->when($request->filled('company_name'), function ($query) use ($request) {
                $query->where('company_name', 'like', '%' . $request->company_name . '%');
            })
            ->orderBy('created_at', 'desc') // Sort by creation date, most recent first
            ->paginate(10); // Adjust the number to fit your needs
    
        return view('external_agencies.index', compact('external_agencies'));
    }
    



    public function create()
    {
        return view('external_agencies.create');
    }

    public function store(Request $request)
{
    $request->validate([
        'company_name' => 'required',
        'license_number' => 'required',
        'license_date' => 'required|date',
        'city' => 'required',
        'country' => 'required',
        'responsible_name' => 'required',
        'email' => 'required|email',
        'phone' => 'required',
        'logo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        'recruitment_fee' => 'required|numeric|min:0',
    ]);

    $logoPath = null;
    if ($request->hasFile('logo')) {
        $logoPath = $request->file('logo')->storeAs(
            'images',
            'logo_' . time() . '.' . $request->file('logo')->extension(),
            'public'
        );
    }

    ExternalAgency::create([
        'company_name' => $request->company_name,
        'license_number' => $request->license_number,
        'license_date' => $request->license_date,
        'city' => $request->city,
        'country' => $request->country,
        'responsible_name' => $request->responsible_name,
        'email' => $request->email,
        'phone' => $request->phone,
        'logo' => $logoPath,
        'recruitment_fee' => $request->recruitment_fee, // Added this line
    ]);

    return redirect()->route('external_agencies.index')->with('success', 'External Agency created successfully.');
}


public function edit(ExternalAgency $agency) // Fixed type hint
{
    return view('external_agencies.edit', compact('agency'));
}

public function update(Request $request, $id)
{
    try {
        $validated = $request->validate([
            'company_name' => 'required|string|max:255',
            'recruitment_fee' => 'required|numeric|min:0',
            'license_number' => 'required|string|max:255',
            'license_date' => 'required|date',
            'city' => 'required|string|max:255',
            'country' => 'required|string|max:255',
            'responsible_name' => 'required|string|max:255',
            'email' => 'required|email|max:255',
            'phone' => 'required|string|max:20',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        $agency = ExternalAgency::findOrFail($id);
        
        // Handle file upload
        if ($request->hasFile('logo')) {
            // Delete old logo if exists
            if ($agency->logo) {
                Storage::disk('public')->delete($agency->logo);
            }
            
            $validated['logo'] = $request->file('logo')->storeAs(
                'agency_logos',
                'logo_'.$agency->id.'_'.time().'.'.$request->file('logo')->extension(),
                'public'
            );
        }

        $agency->update($validated);

        return response()->json([
            'success' => true,
            'message' => 'External Agency updated successfully'
        ]);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'An error occurred while updating the agency: ' . $e->getMessage()
        ], 500);
    }
}


    public function destroy($id)
    {
        try {
            // Find the external agency by its ID
            $agency = ExternalAgency::findOrFail($id);

            // Delete the agency
            $agency->delete();

            // Redirect back with a success message
            return redirect()->route('external_agencies.index')->with('success', 'External Agency deleted successfully.');
        } catch (\Illuminate\Database\QueryException $e) {
            // Check if it's a foreign key constraint violation
            if ($e->getCode() == 23000) {
                return redirect()->route('external_agencies.index')
                    ->with('error', 'Cannot delete this agency because it has associated records (e.g., invoices). Please remove the associated records first.');
            }
            
            // For other database errors
            return redirect()->route('external_agencies.index')
                ->with('error', 'An error occurred while deleting the agency. Please try again.');
        } catch (\Exception $e) {
            // For any other unexpected errors
            return redirect()->route('external_agencies.index')
                ->with('error', 'An unexpected error occurred. Please try again.');
        }
    }

}
