<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\Worker;
use App\Models\Invoice;
use App\Models\Application;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;

class ReportController extends Controller
{
    public function index()
    {
        return view('reports.generate');
    }

    public function generate(Request $request)
    {
        $reportType = $request->input('report_type');
        $dateRange = $request->input('date_range');
        $format = $request->input('format', 'pdf');

        // Parse date range
        $dates = explode(' - ', $dateRange);
        $startDate = $dates[0] ?? null;
        $endDate = $dates[1] ?? null;

        // Get data based on report type
        $data = match($reportType) {
            'users' => $this->getUsersReport($startDate, $endDate),
            'workers' => $this->getWorkersReport($startDate, $endDate),
            'invoices' => $this->getInvoicesReport($startDate, $endDate),
            'applications' => $this->getApplicationsReport($startDate, $endDate),
            default => collect([])
        };

        if ($data->isEmpty()) {
            return back()->with('error', 'No data found for the selected criteria.');
        }

        // Generate report in requested format
        return match($format) {
            'pdf' => $this->generatePdf($data, $reportType),
            'csv' => $this->generateCsv($data, $reportType),
            'xlsx' => $this->generateExcel($data, $reportType),
            default => back()->with('error', 'Invalid export format.')
        };
    }

    private function getUsersReport($startDate, $endDate)
    {
        $query = User::query();
        if ($startDate && $endDate) {
            $query->whereBetween('created_at', [$startDate, $endDate]);
        }
        return $query->get();
    }

    private function getWorkersReport($startDate, $endDate)
    {
        $query = Worker::query();
        if ($startDate && $endDate) {
            $query->whereBetween('created_at', [$startDate, $endDate]);
        }
        return $query->get();
    }

    private function getInvoicesReport($startDate, $endDate)
    {
        $query = Invoice::query();
        if ($startDate && $endDate) {
            $query->whereBetween('created_at', [$startDate, $endDate]);
        }
        return $query->get();
    }

    private function getApplicationsReport($startDate, $endDate)
    {
        $query = Application::query();
        if ($startDate && $endDate) {
            $query->whereBetween('created_at', [$startDate, $endDate]);
        }
        return $query->get();
    }

    private function generatePdf($data, $reportType)
    {
        $pdf = PDF::loadView("reports.{$reportType}-pdf", ['data' => $data]);
        return $pdf->download("{$reportType}_report.pdf");
    }

    private function generateCsv($data, $reportType)
    {
        return Excel::download(new ReportsExport($data), "{$reportType}_report.csv");
    }

    private function generateExcel($data, $reportType)
    {
        return Excel::download(new ReportsExport($data), "{$reportType}_report.xlsx");
    }
} 