<?php

namespace App\Http\Controllers;

use App\Models\Visitor;
use Illuminate\Http\Request;
use App\Models\VisitorLog;

class VisitorController extends Controller
{

    
    public function index(Request $request)
    {
        // Visitor Query
        $query = Visitor::query();
        $logQuery = VisitorLog::with('visitor')->latest();
  
        // Apply visitor filters (Checked-in, Checked-out, Deleted)
        if ($request->has('filter')) {
            if ($request->filter === 'checked-in') {
                $query->whereNull('check_out');
            } elseif ($request->filter === 'checked-out') {
                $query->whereNotNull('check_out');
            } elseif ($request->filter === 'deleted') {
                $query->onlyTrashed(); 
            }
        }

        // Search Visitors
        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%$search%")
                  ->orWhere('email', 'like', "%$search%")
                  ->orWhere('phone', 'like', "%$search%")
                  ->orWhere('purpose', 'like', "%$search%");
            });
        }

        // Filter Visitors (Checked-in, Checked-out, Deleted)
        if ($request->has('filter')) {
            if ($request->filter === 'checked-in') {
                $query->whereNull('check_out');
            } elseif ($request->filter === 'checked-out') {
                $query->whereNotNull('check_out');
            } elseif ($request->filter === 'deleted') {
                $query->onlyTrashed(); // This will now work if SoftDeletes is enabled in the model
            }
        }

        // Search Logs
        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $logQuery->where(function ($q) use ($search) {
                $q->whereHas('visitor', function ($subQuery) use ($search) {
                    $subQuery->where('name', 'like', "%$search%");
                })->orWhere('action', 'like', "%$search%")
                  ->orWhereDate('action_time', $search);
            });
        }

        // Filter Logs (Check-ins or Check-outs)
        if ($request->has('log_filter') && !empty($request->log_filter)) {
            $logQuery->where('action', $request->log_filter);
        }

        // Get Visitors
        $visitors = $query->latest()->paginate(10);
        $visitorLogs = $logQuery->paginate(10);

        // Get counts for each category of visitors
        $allCount = Visitor::count();
        $checkedInCount = Visitor::whereNull('check_out')->count();
        $checkedOutCount = Visitor::whereNotNull('check_out')->count();
        $deletedCount = Visitor::onlyTrashed()->count();

        return view('visitors.index', compact('visitors', 'visitorLogs', 'checkedInCount', 'checkedOutCount', 'deletedCount', 'allCount'));
    }

    public function create()
    {
        return view('visitors.create');
    }

    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email',
            'phone' => 'nullable|string|max:20',
            'purpose' => 'required|string|max:255',
        ]);

        $visitor = Visitor::create([
            'name' => $request->name,
            'email' => $request->email,
            'phone' => $request->phone,
            'purpose' => $request->purpose,
            'check_in' => now(),
        ]);

        // Log the check-in action
        VisitorLog::create([
            'visitor_id' => $visitor->id,
            'action' => 'Checked In',
            'action_time' => now(),
        ]);

        return redirect()->route('visitors.index')->with('success', 'Visitor checked in successfully.');
    }

    public function checkout($id)
    {
        $visitor = Visitor::findOrFail($id);
        
        // If already checked out, prevent updating again
        if ($visitor->check_out) {
            return redirect()->route('visitors.index')->with('error', 'Visitor has already checked out.');
        }

        // Check-out the visitor
        $visitor->check_out = now();
        $visitor->save();

        // Log the check-out action
        VisitorLog::create([
            'visitor_id' => $visitor->id,
            'action' => 'Checked Out',
            'action_time' => now(),
        ]);

        return redirect()->route('visitors.index')->with('success', 'Visitor checked out successfully.');
    }

    public function show(Visitor $visitor)
    {
        return view('visitors.show', compact('visitor'));
    }

    public function edit(Visitor $visitor)
    {
        return view('visitors.edit', compact('visitor'));
    }

    public function update(Request $request, Visitor $visitor)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'nullable|email',
            'phone' => 'nullable|string|max:15',
            'purpose' => 'nullable|string|max:500',
        ]);

        $visitor->update($request->all());

        return redirect()->route('visitors.index')->with('success', 'Visitor updated successfully.');
    }

    public function destroy(Visitor $visitor)
    {
        // Log the deletion action before deleting
        VisitorLog::create([
            'visitor_id' => $visitor->id,
            'action' => 'Deleted',
            'action_time' => now(),
        ]);
        
        // Delete the visitor record
        $visitor->delete();
    
        return redirect()->route('visitors.index')->with('success', 'Visitor deleted successfully.');
    }
}
