<?php

namespace App\Http\Controllers;

use App\Models\SubAgent;
use App\Models\Voucher;
use App\Models\Worker;
use App\Models\EmploymentStatus;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use PDF;

class VoucherController extends Controller
{
    /**
     * Display a listing of the vouchers.
     */
    public function index(Request $request)
    {
        // Get workers with 'travelled' status
        $travelledWorkers = Worker::whereHas('employmentStatus', function($query) {
                $query->where('status', 'travelled');
            })
            ->orderBy('surname')
            ->get();
    
        // Query vouchers with filters
        $vouchers = Voucher::with(['subAgent', 'workers'])
            ->when($request->filled('search'), function($query) use ($request) {
                $search = $request->input('search');
                return $query->where(function($q) use ($search) {
                    $q->where('voucher_number', 'like', "%{$search}%")
                      ->orWhereHas('worker', function($q) use ($search) {
                          $q->where('surname', 'like', "%{$search}%")
                            ->orWhere('given_names', 'like', "%{$search}%");
                      })
                      ->orWhereHas('subAgent', function($q) use ($search) {
                          $q->where('name', 'like', "%{$search}%");
                      });
                });
            })
            ->latest()
            ->paginate(10);
    
        return view('vouchers.index', [
            'vouchers' => $vouchers,
            'travelledWorkers' => $travelledWorkers
        ]);
    }
    /**
     * Show the form for creating a new voucher.
     */
    public function create($workerId = null)
{
    // Get the currently selected sub-agent (from session or request)
    $selectedSubAgentId = session('selected_sub_agent_id') ?? request('sub_agent_id');

    // Get workers with 'travelled' status for the selected sub-agent
    $workers = Worker::whereHas('employmentStatus', function($query) {
            $query->where('status', 'travelled');
        })
        ->when($selectedSubAgentId, function($query) use ($selectedSubAgentId) {
            $query->where('sub_agent_id', $selectedSubAgentId);
        })
        ->orderBy('surname')
        ->get();

    // Get selected worker if ID provided
    $selectedWorker = null;
    if ($workerId) {
        $selectedWorker = $workers->firstWhere('id', $workerId);
    }

    $subAgents = SubAgent::all();

    return view('vouchers.create', [
        'workers' => $workers,
        'selectedWorker' => $selectedWorker,
        'subAgents' => $subAgents,
        'selectedSubAgentId' => $selectedSubAgentId
    ]);
}
    /**
     * Store a newly created voucher in storagzz.
     */
    public function store(Request $request)
    {
        try {
            Log::info('Voucher creation request:', $request->all());

            $validator = \Validator::make($request->all(), [
                'sub_agent_id' => 'required|exists:sub_agents,id',
                'worker_id' => [
                    'required',
                    'exists:workers,id',
                    function ($attribute, $value, $fail) {
                        $worker = Worker::with('employmentStatus')->find($value);
                        if (!$worker->employmentStatus || $worker->employmentStatus->status !== 'travelled') {
                            $fail('The selected worker is not eligible for voucher creation.');
                        }
                    }
                ],
                'voucher_type' => 'required|in:' . implode(',', [
                    Voucher::TYPE_ADVANCE,
                    Voucher::TYPE_FINAL
                ]),
                'amount' => 'required|numeric|min:0',
                'notes' => 'nullable|string|max:1000'
            ]);

            if ($validator->fails()) {
                Log::error('Validation failed:', $validator->errors()->toArray());
                return back()
                    ->withErrors($validator)
                    ->withInput();
            }

            DB::beginTransaction();

            // Create voucher
            $voucher = Voucher::create([
                'sub_agent_id' => $request->sub_agent_id,
                'worker_id' => $request->worker_id,
                'voucher_type' => $request->voucher_type,
                'total_amount' => $request->amount,
                'status' => Voucher::STATUS_PENDING,
                'notes' => $request->notes,
                'created_by' => auth()->id()
            ]);

            Log::info('Voucher created:', $voucher->toArray());

            // Update worker status if this is a final payment
            if ($request->voucher_type === Voucher::TYPE_FINAL) {
                EmploymentStatus::updateOrCreate(
                    ['worker_id' => $request->worker_id],
                    ['status' => 'completed']
                );
            }

            DB::commit();

            return redirect()
                ->route('vouchers.show', $voucher)
                ->with('success', 'Voucher created successfully.');

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error creating voucher: ' . $e->getMessage());
            Log::error($e->getTraceAsString());

            return back()
                ->with('error', 'Failed to create voucher: ' . $e->getMessage())
                ->withInput();
        }
    }

    /**
     * Display the specified voucher.
     */
    public function show(Voucher $voucher)
    {
        $voucher->load(['subAgent', 'workers.employmentStatus']);
        return view('vouchers.show', compact('voucher'));
    }

    /**
     * Update the specified voucher's status.
     */
    public function updateStatus(Request $request, Voucher $voucher)
    {
        $request->validate([
            'status' => 'required|in:' . implode(',', [
                Voucher::STATUS_APPROVED,
                Voucher::STATUS_REJECTED,
                Voucher::STATUS_PAID
            ]),
            'rejection_reason' => 'required_if:status,'.Voucher::STATUS_REJECTED.'|nullable|string|max:500'
        ]);

        try {
            DB::beginTransaction();

            $voucher->update([
                'status' => $request->status,
                'rejection_reason' => $request->rejection_reason,
                'processed_by' => auth()->id(),
                'processed_at' => now()
            ]);

            // Additional actions based on status
            if ($request->status === Voucher::STATUS_PAID) {
                $voucher->update(['paid_at' => now()]);
                
                // Optionally update worker status when payment is made
                if ($voucher->voucher_type === Voucher::TYPE_FINAL) {
                    EmploymentStatus::updateOrCreate(
                        ['worker_id' => $voucher->worker_id],
                        ['status' => 'payment_completed']
                    );
                }
            }

            DB::commit();

            return back()->with('success', 'Voucher status updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error updating voucher status: ' . $e->getMessage());
            return back()->with('error', 'Failed to update voucher status.');
        }
    }

    /**
     * Remove the specified voucher from storage.
     */
    public function destroy(Voucher $voucher)
    {
        if (!$voucher->canBeEdited()) {
            return back()->with('error', 'This voucher cannot be deleted.');
        }

        try {
            DB::beginTransaction();
            
            $voucher->delete();
            
            DB::commit();
            
            return redirect()
                ->route('vouchers.index')
                ->with('success', 'Voucher deleted successfully.');
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error deleting voucher: ' . $e->getMessage());
            return back()->with('error', 'Failed to delete voucher.');
        }
    }

    /**
     * Export voucher as PDF.
     */
    public function export(Voucher $voucher)
    {
        $voucher->load(['subAgent', 'worker.employmentStatus']);
        $pdf = PDF::loadView('vouchers.pdf', compact('voucher'));
        
        return $pdf->download("voucher_{$voucher->voucher_number}.pdf");
    }
}