<?php

namespace App\Services;

use App\Models\Setting;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Validator;

class SettingsService
{
    protected $validationRules = [
        'general' => [
            'site_name' => 'required|string|max:255',
            'site_description' => 'required|string|max:1000',
        ],
        'backup' => [
            'backup_frequency' => 'required|in:daily,weekly,monthly',
            'backup_retention' => 'required|integer|min:1|max:365',
        ],
        'notifications' => [
            'email_notifications' => 'required|in:enabled,disabled',
            'notification_email' => 'required|email|max:255',
        ],
        'security' => [
            'session_timeout' => 'required|integer|min:5|max:1440',
            'max_login_attempts' => 'required|integer|min:1|max:10',
        ],
    ];

    public function validateSettings(array $settings)
    {
        $errors = [];
        foreach ($settings as $key => $value) {
            $group = $this->getGroupFromKey($key);
            if (isset($this->validationRules[$group][$key])) {
                $validator = Validator::make(
                    [$key => $value],
                    [$key => $this->validationRules[$group][$key]]
                );

                if ($validator->fails()) {
                    $errors[$key] = $validator->errors()->first();
                }
            }
        }

        return $errors;
    }

    public function updateSettings(array $settings)
    {
        $errors = $this->validateSettings($settings);
        if (!empty($errors)) {
            return ['success' => false, 'errors' => $errors];
        }

        try {
            foreach ($settings as $key => $value) {
                $group = $this->getGroupFromKey($key);
                $type = $this->getTypeFromKey($key);
                Setting::set($key, $value, $type, $group);
            }

            $this->clearCache();
            return ['success' => true];
        } catch (\Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }

    public function getSettings()
    {
        return Cache::remember('app_settings', 3600, function () {
            return Setting::all()->groupBy('group')->map(function ($group) {
                return $group->pluck('value', 'key');
            })->toArray();
        });
    }

    protected function getGroupFromKey($key)
    {
        $parts = explode('.', $key);
        return $parts[0] ?? 'general';
    }

    protected function getTypeFromKey($key)
    {
        $numericKeys = ['backup_retention', 'session_timeout', 'max_login_attempts'];
        return in_array($key, $numericKeys) ? 'integer' : 'string';
    }

    protected function clearCache()
    {
        Cache::forget('app_settings');
    }
} 